// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use select with multiple resources
-----------------------------------------

A select statement waits for one of a set of inputs to become ready,
performs the selected input and then executes a corresponding body of
code. Each input is proceeded by the keyword case and its body must be
terminated with a break or return statement. Case statements are not
permitted to contain output operations. Select statements can be used
to tie together inputs from multiple resources, in this example we
look at combining timers, channels and ports into a single select
statement.
**/


#include <xs1.h>
#include <platform.h>
#include <print.h>

on tile[0] : in port port_a = XS1_PORT_1A;

void RX (chanend chnlend_a, chanend chnlend_b)
{
  unsigned chnl_input_a = 0;
  unsigned chnl_input_b = 0;
  unsigned port_input_a = 0;
  timer tmr_a;
  unsigned start_time = 0;

  tmr_a :> start_time;

  while (1)
  {
/**
    The select statement implemented below is used to wait for either an
    input on chnlend_a, an input on chnlend_b, an input from port_a or a
    timer value. When an input is received from any of the cases an
    message is printed to display which case has fired. The timer case can
    be used to wake up and execute at a particular rate, in this example
    it is every 10000 timer cycles.
**/

    select
    {
      case chnlend_a :>  chnl_input_a :
        printstr("Channel Input A Received ");
        printintln(chnl_input_a);
        break;
      case chnlend_b :> chnl_input_b :
        printstr("Channel Input B Received ");
        printintln(chnl_input_b);
        break;
      case port_a when pinseq(1) :> port_input_a:
        printstr("Port Input A Recieved ");
        printintln(port_input_a);
        break;
      case tmr_a when timerafter(start_time + 10000) :> start_time:
        printstrln("Timer A Fired ");
        break;
      }
    }

}

void TX (chanend chnlend_a, chanend chnlend_b)
{
  unsigned chnl_output = 0;

  while (1)
  {
    chnl_output++;

    //if the value of chnl_output is divisible by 5 then output it to chnlend_a
    if (chnl_output % 5 == 0)
    {
      chnlend_a <: chnl_output;
    }

    //if the value of chnl_output is divisible by 10 then output it to chnlend_b
    if (chnl_output % 10 == 0)
    {
      chnlend_b <: chnl_output;
    }
  }
}

int main(void)
{
  chan chnl_a, chnl_b;

  par
  {
    on stdcore[0] : RX(chnl_a, chnl_b);
    on stdcore[1] : TX(chnl_a, chnl_b);
  }

  return 0;
}

