// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use a select function
----------------------------

Select statements can be implemented as functions, allowing their
reuse in different contexts.
**/


#include <platform.h>
#include <print.h>
#include <xs1.h>

on stdcore[0] : in port port_in = XS1_PORT_1A;

/**
    The declaration
**/


select channel_input(chanend c_a, chanend c_b)

/**
declares 'channel_input' to be a select function that takes two
arguments 'c_a' and 'c_b'. The body of the select function adheres to
the  same rules as the select statement.
**/


{
  case c_a :> int chnl_input_a :
    printstr("Channel Input A Received ");
    printintln(chnl_input_a);
    break;
  case c_b :> int chnl_input_b :
    printstr("Channel Input B Received ");
    printintln(chnl_input_b);
    break;
}


void do_port_input (unsigned &port_input_data)
{
  printstr("Port Input received ");
  printintln(port_input_data);
}

void RX (chanend c_a, chanend c_b)
{
  unsigned port_input_data = 0;

  while (1)
  {
/**
    You can call the select function from the relevant location in the
    program.  In this example the select function is nested within a
    select statement where it is inputting data from either one of the two
    channel ends 'c_a, c_b' or from the input port 'port_in'
**/


    select {
      case channel_input(c_a, c_b);
      case port_in when pinsneq(port_input_data) :> port_input_data :
        do_port_input(port_input_data);
        break;
    }

  }
}

void TX (chanend c_a, chanend c_b)
{
  unsigned chnl_output = 0;

  while (1)
  {
    chnl_output++;

    //if the value of chnl_output is divisible by 5 then output it to chnlend_a
    if (chnl_output % 5 == 0)
    {
      c_a <: chnl_output;
    }

    //if the value of chnl_output is divisible by 10 then output it to chnlend_b
    if (chnl_output % 10 == 0)
    {
      c_b <: chnl_output;
    }
  }
}

int main(void)
{
  chan chnl_a, chnl_b;

  par
  {
    on stdcore[0] : RX(chnl_a, chnl_b);
    on stdcore[1] : TX(chnl_a, chnl_b);
  }

  return 0;
}

