// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use the select statement with a replicator
-------------------------------------------------

A replicator can be used within a select statement.  It is most useful
at the top-level select statement where multiple calls to a select
function are required.

In this example, 4 bits are expected to be read from each of the
defined in ports. XS1_PORT_1A and XS1_PORT_1B will form byte[0] and
XS1_PORT_1C and XS1_PORT_1D will form byte[1].
**/


#include <platform.h>
#include <print.h>
#include <xs1.h>

#define NUM_BYTES 2

in port port_input1[2] = { XS1_PORT_1A,
                           XS1_PORT_1C };

in port port_input2[2] = { XS1_PORT_1B,
                           XS1_PORT_1D };

/**
    The select function 'in_bit' performs the input operation for one of
    the two ports as defined by the parameters 'p0' and 'p1'
**/


select in_bit(in port p0, in port p1, char & byte)
{
  case p0 :> int p0_data :
    if (p0_data == 1)
      byte = (byte << 1) | 1;
    break;
  case p1 :> int p1_data :
     if (p1_data == 1)
       byte = (byte << 1) | 0;
    break;
}


int main(void)
{
  char byte[NUM_BYTES] = {0, 0};

  for (int i = 0; i < 16; ++i)
  {

/**
The top level select statement uses the replicator '(int j = 0; j <
NUM_BYTES; ++j)' to make multiple calls to the select function
'in_bit'.  Ports XS1_PORT_1A or XS1_PORT_1B are input in the first
iteration of the replicator.  Ports XS1_PORT_1C or XS1_PORT_1D are
input on the second iteration.
**/


    select
    {
      case (int j = 0; j < NUM_BYTES; ++j)
          in_bit(port_input1[j], port_input2[j], byte[j]);
    }

/**
Using the replicator simplifies the code, making it much easier to
read. The number of iterations need not be constant but the iterator
must not be modified outside of the replicator.
**/


  }

  printstr("Value of byte[0] is ");
  printhexln(byte[0]);
  printstr("Value of byte[1] is ");
  printhexln(byte[1]);

  return 0;
}


