// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use the select statement with a guard
--------------------------------------------

A select statement waits for one of a set of inputs to become ready,
performs the selected input and then executes a corresponding body of
code. A guard can be used within a select case to enable or disable
the input operation.
**/



#include <platform.h>
#include <print.h>


void RX(chanend chnlend_a, chanend chnlend_b)
{

/**
In this example a guard is used to control input between 'chnlend_a'
and 'chnlend_b'.  The initial state is set so that input from
'chnlend_a' is enabled and input from 'chnlend_b' is disabled.
**/


  int is_a = 1;
  int is_b = 0;


  int counter = 0;

  while (1)
  {

/**
The select body receives 10 inputs on 'chnlend_a' at which point
'chnlend_a' is disabled and 'chnlend_b' is enabled to perform the same
input sequence.  The program continues to alternate receiving from the
two inputs.
**/


    select
    {
      case is_a => chnlend_a :> int chnl_input_a :
        printstr("Channel Input A Received ");
        printintln(chnl_input_a);
        if (++counter == 10)
        {
          is_a = 0;
          is_b = 1;
          counter = 0;
        }
        break;
      case is_b => chnlend_b :> int chnl_input_b :
        printstr("Channel Input B Received ");
        printintln(chnl_input_b);
        if (++counter == 10)
        {
          is_a = 1;
          is_b = 0;
          counter = 0;
        }
        break;
    }


  }
}

void TX_A(chanend chnlend_a)
{
  unsigned chnl_output = 0;

  while (1)
  {
    chnlend_a <: chnl_output++;
  }
}

void TX_B(chanend chnlend_b)
{
  unsigned chnl_output = 0;

  while (1)
  {
    chnlend_b <: chnl_output++;
  }
}

int main(void)
{
  chan chnl_a, chnl_b;

  par
  {
    on tile[0] : RX(chnl_a, chnl_b);
    on tile[1] : TX_A(chnl_a);
    on tile[1] : TX_B(chnl_b);
  }

  return 0;
}

