// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use the select statement with a default case
---------------------------------------------------

A select statement waits for one of a set of inputs to become ready,
performs the selected input and then executes a corresponding body of
code. There are however occasions when it is not reasonable to wait on
one of the inputs to become ready.  In this instance a default case
statement can be used to exit the select statement immediately if no
inputs are ready.
**/


#include <platform.h>
#include <print.h>


void RX (chanend chnlend_a, chanend chnlend_b)
{
  while (1)
  {
/**
    In this example the select statement is used to wait for either an
    input on 'chnlend_a' or an input on 'chnlend_b'.  If an input is
    available from either channel the value of the input is printed.  If
    neither channel has an input ready then the default case is executed.
**/


    select
    {
      case chnlend_a :> int chnl_input_a :
        printstr("Channel Input A Received ");
        printintln(chnl_input_a);
        break;
      case chnlend_b :> int chnl_input_b :
        printstr("Channel Input B Received ");
        printintln(chnl_input_b);
        break;
      default :
        printstrln("Default Case executed");
        break;
    }

/**
    Care should be taken when using a default case statement as it can
    have the same effect as the processor polling for input.  This will
    ultimately have a drain on the power and waste valuable clock cycles.
**/

  }
}

void TX (chanend chnlend_a, chanend chnlend_b)
{
  unsigned time;
  timer t;
  unsigned chnl_output = 0;

  t :> time;

  while (1)
  {
    chnl_output++;

    //if the value of chnl_output is divisible by 5 then output it to chnlend_a
    if (chnl_output % 5 == 0)
    {
      chnlend_a <: chnl_output;
    }

    //if the value of chnl_output is divisible by 10 then output it to chnlend_b
    if (chnl_output % 10 == 0)
    {
      chnlend_b <: chnl_output;
    }

    //use a timer to delay the TX so the default case has a chance to execute.
    time += 500;
    t when timerafter(time) :> void;
  }
}

int main(void)
{
  chan chnl_a, chnl_b;

  par
  {
    on tile[0] : RX(chnl_a, chnl_b);
    on tile[1] : TX(chnl_a, chnl_b);
  }

  return 0;
}

