// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use buffering for port output
------------------------------------

You can set a port into *buffered output* port. This tells the XMOS
architecture to set the port into a mode that serializes the data. Any
outputs will be places into a small fifo (the *transfer register*) and
writing to the port will fill this fifo. The fifo will then be output
independently at the clock rate of the port. The size of the transfer
register is user specified and does not need to be the same as the
port width.

This mode can improve perfomance since it allows outputs to be driven
to the pins independently to the instructions of the program.
**/


#include <xs1.h>

/**
    The declaration
**/


out buffered port:32 out_port = XS1_PORT_1A;

/**
declares a buffered output port named 'out_port', which refers to the
1-bit port identifier XS1_PORT_1A. Note that the size of the port FIFO
is specified as part of the buffered port declaration using the colon.
The allowable sizes for this FIFO on XS1 devices are: 1, 4, 8 and 32.
**/


//a clock block is required to provide edges for the port output
clock clk = XS1_CLKBLK_1;

int main(void) {
  int counter = 0;
  configure_clock_rate(clk, 100, 8);
  configure_out_port(out_port, clk, 0);
  start_clock(clk);

  while(counter < 1000) {
    /** Output to the port is performed as normal.
     **/
    out_port <: counter;
  /** The port buffers the data so that the processor can continue executing
      subsequent instructions.  On each falling edge of the clock, the port
      takes the next bit of data from its buffer and drives it on its pins.
   **/
    ++counter;
  }

  return 0;
}
