// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use buffering for port input
-----------------------------------

You can set a port into *buffered input* port. This tells the XMOS
architecture to set the port into a mode that de-serializes the data.
Any inputs will be places into a small fifo (the *transfer register*)
and reading the port will read out the fifo when it is full. The size
of the transfer register is user specified and does not need to be the
same as the port width.

This mode can improve perfomance since it allows inputs to be sampled
off the pins independently to the instructions of the program.
**/


#include <xs1.h>

/**
    The declaration
**/


in buffered port:32 in_port = XS1_PORT_1A;

/**
declares a buffered input port named 'in_port', which refers to the
1-bit port identifier XS1_PORT_1A. Note that the size of the port FIFO
is specified as part of the port declaration using the colon. The
allowable sizes for this FIFO on XS1 devices are: 1, 4, 8 and 32.
**/


clock clk = XS1_CLKBLK_1;

int main(void) {
  int data;
  configure_clock_rate(clk, 100, 8);
  configure_in_port(in_port, clk);
  start_clock(clk);

  while(1) {
    /** Input from the port is performed as normal.
     **/
    in_port :> data;
/**
    At most one value is sampled by the port and inserted to the FIFO per
    period of its clock. If the FIFO is full, the oldest value is dropped
    to make room for the most recently sampled value.
**/

  }

  return 0;
}

