// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to use overlays and libflash in the same application
--------------------------------------------------------

When the flash overlay runtime is initialized it takes ownership of
the xCORE ports used to access the SPI flash device. This prevents
them being used for any other purpose (e.g. reading and writing data
using libflash).
**/


#include <platform.h>
#include <xs1.h>
#include <overlay_flash.h>
#include <flash.h>

fl_SPIPorts spi_ports = {
  PORT_SPI_MISO,
  PORT_SPI_SS,
  PORT_SPI_CLK,
  PORT_SPI_MOSI,
  XS1_CLKBLK_1
};

fl_SPIPorts * movable spi_ports_ptr = &spi_ports;

[[overlay]]
void first()
{
  // Put code for first overlay here...
}

[[overlay]]
void second()
{
  // Put code for second overlay here...
}

int main()
{
  // Initialise the flash overlay runtime.
  overlay_flash_init(move(spi_ports_ptr), 100, 8);

/**
  If necessary an application can ask the flash overlay runtime to
  transfer ownership of SPI ports back to the application:
**/

  fl_SPIPorts * movable spi_ports = overlay_flash_claim_ports();
/**
  If the overlay runtime is in the middle of loading an overlay the call
  to 'overlay_flash_claim_ports' will block until it has finished
  loading the overlay.
  
  Once the application has ownership of the SPI ports it can use them to
  read and write data from the flash using libflash:
**/

  fl_connect(*spi_ports);
  // Access flash here...
  fl_disconnect();

/**
  While the application has ownership of the ports a call to function in
  an overlay that is not loaded will block. Call
  'overlay_flash_return_ports' to give ownership of the ports back to
  the overlay runtime:
**/

  overlay_flash_return_ports(move(spi_ports));

  first();
  second();

  return 0;
}
