// Copyright (c) 2015, XMOS Ltd, All rights reserved
/**
How to implement a fair select
------------------------------

A select statement waits for one of a set of inputs to become ready.
If more than one input is ready the select statement arbitrarily
chooses one to service. Since the choice of which input to service is
made arbitrarily, no guarantee of fairness is provided.
**/


#include <platform.h>
#include <print.h>
#include <stdlib.h>

#define NUM_CLIENTS 6
#define NUM_MESSAGES 50

void simple_select(chanend c[NUM_CLIENTS])
{
/**
Consider the following code which handles incoming messages from
multiple clients using a select statement in a loop:
**/

  unsigned received = 0;
  while (1) {
    select {
    case (unsigned i = 0; i < NUM_CLIENTS; i++) c[i] :> int x:
      printstr("Received message from client ");
      printintln(i);
      received++;
      if (received == NUM_MESSAGES)
        return;
      break;
    }
  }
}

/**
If a client sends a message while the server is handling a message for
another client, the output statement in that client task will block.
If the other clients are repeatedly sending messages to the server
then, since the select is not fair, the server may never get round to
handling the message from this client.
**/


void fair_select(chanend c[NUM_CLIENTS])
{
/**
To fix this problem we can use the combination of boolean guards on
select cases and the default case to implement a fair server that
gives equal priority to all its clients.
**/

  int disabled[NUM_CLIENTS] = {0};
  int default_enabled = 0;
  unsigned received = 0;
  while (1) {
    select {
    case (unsigned i = 0; i < NUM_CLIENTS; i++) !disabled[i] => c[i] :> int x:
      printstr("Received message from client ");
      printintln(i);
      disabled[i] = 1;
      default_enabled = 1;
      received++;
      if (received == NUM_MESSAGES)
        return;
      break;
    default_enabled => default:
      for (unsigned i = 0; i < NUM_CLIENTS; i++) {
        disabled[i] = 0;
      }
      default_enabled = 0;
      break;
    }
  }
/**
After an event on a channel end has been received, events on that
channel end are disabled. This ensures that the server doesn't accept
two events from the same channel end without handling events on other
channel ends that are ready to event first.

Each time an event is serviced, the number of channel ends that have
events enabled is reduced by one. Eventually we will run out of
channel ends that are both ready and have events enabled. At this
point the default case in the select will be taken and events will be
re-enabled for all of the channel ends.

The fair server satisfies the property that if an client sends a
message to the server, the server will handle at most one message from
every other client before handling the message from this client. This
provides a minimum guarantee of responsiveness to each client
regardless of what the other clients are doing.
**/

}

static void client_task(chanend c)
{
  while (1) {
    c <: 0;
  }
}

int main(void)
{
  chan c[NUM_CLIENTS];

  par
  {
    {
      printstr("Simple select:\n");
      simple_select(c);
      printstr("Fair select:\n");
      fair_select(c);
      _Exit(1);
    }
    par (unsigned i = 0; i < NUM_CLIENTS; i++) {
      client_task(c[i]);
    }
  }

  return 0;
}
