// Copyright (c) 2016, XMOS Ltd, All rights reserved
#include <xs1.h>
#include <print.h>
/**
How to define and use a distributable function
----------------------------------------------
**/


interface wiggle_if {
  void wiggle(int n);
  void finish();
};

/**
If a task is a never-ending loop containing a single select (like a
combinable function) that *only has cases responding to interface
messages*, the function can be marked as *distributable*. For example:
**/


[[distributable]]
void port_wiggler(server interface wiggle_if c, port p)
{
  // This task waits for a message on the interface c and
  // wiggles the port p the required number of times.
  while (1) {
    select {
    case c.wiggle(int n):
      printstrln("Wiggling port.");
      for (int i=0;i<n;i++) {
        p <: 1;
        p <: 0;
      }
      break;
    case c.finish():
      return;
    }
  }
}


void task1(client interface wiggle_if c)
{
  c.wiggle(5);
}

port p = XS1_PORT_1A;

/**
 A distributable task can be distributed within a 'par'. This means
 that the task will not run on any particular core but will be run on
 the core of the task that calls to it.
**/


int main() {
  interface wiggle_if c;
  par {
    task1(c);
    [[distribute]] port_wiggler(c, p);
  }
  return 0;
}

/**
 A distributed task must be on the same tile as the tasks it is
 connected to.
**/

