// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <platform.h>

/**
How to use the xSCOPE state machine event type
----------------------------------------------

xSCOPE is fully supported on hardware platforms which provide an XMOS
link between the target device and the XSYS development connector, is
is also supported via xSIM.

View the document (http://www.xmos.com/doc/XM-000957-PC/latest/page31
#trace-data-with-xscope) for further information on tracing data from
XMOS applications.

This example provides a simple demonstration of using the xSCOPE state
machine event type for data logging from within an xCORE application.
The state machine event type is used for recording the value of a
variables associated with state change within an application. This
allows a user to see how long the device has spent in specific states
for timing and debugging purposes.

The probe configuration is handled by the user providing a
config.xscope file which is picked up as part of the application
build.

This example assumes you are familiar with creating a run
configuration and enabling the associated xSCOPE options in that run
configuration in xTIMEcomposer Studio or using the command line tools.
**/


/**
    In order to used xSCOPE the correct header file must be included in
    the application
**/

#include <xscope.h>

void wait(int delay)
{
  timer t;
  int i;
  t:>i;
  t when timerafter(i + delay):>i;
}

/**
    Declare a variable you are going to trace as an xSCOPE state machine
    event type
**/

unsigned int current_state = 1;

void change_state(unsigned int new_state)
{
  current_state = new_state;
  /** The xscope_probe_data() function is used to send the contents of user
      variable current_state to xSCOPE probe STATE_TRANSITIONS for logging
   **/
  xscope_int(STATE_TRANSITIONS, current_state);

  switch (new_state) {
  case 1:
    wait(10000);
    break;
  case 2:
    wait(40000);
    break;
  case 3:
    wait(70000);
    break;
  case 4:
    wait(100000);
    break;
  case 5:
    wait(20000);
    break;
  default:
    break;
  }
}



int main(void)
{
  par {
    on tile[0]: {
      for (int i = 0; i < 5; i++) {
        change_state(2);
        change_state(3);
        change_state(4);
        change_state(3);
        change_state(5);
        change_state(1);
        change_state(3);
        change_state(2);
        change_state(1);
        change_state(4);
      }
    }
  }

  return 0;
}
