// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <xs1.h>
#include <platform.h>
#include <flashlib.h>
#include <timer.h>

/*
 * the patterns for each LED on startKIT are:
 *   0x80000 0x40000 0x20000
 *   0x01000 0x00800 0x00400
 *   0x00200 0x00100 0x00080
 *
 * As the leds go to 3V3, 0x00000 drives all 9 leds on,
 * and 0xE1F80 drives all nine leds off.
 * The 16 bytes stored in the data partition
 * drive a dash, backslash, pipe, and slash.
 * 0x000E0380
 * 0x00061700
 * 0x000A1680
 * 0x000C1580
 */

// Ports for SPI access on startKIT.
fl_SPIPorts ports = {
  PORT_SPI_MISO,
  PORT_SPI_SS,
  PORT_SPI_CLK,
  PORT_SPI_MOSI,
  on tile[0]: XS1_CLKBLK_1
};

// Port where the leds reside on startKIT.
port p32 = XS1_PORT_32A;

// List of SPI devices that are supported by default.
fl_DeviceSpec deviceSpecs[] =
{
  FL_DEVICE_ATMEL_AT25DF041A,
  FL_DEVICE_ST_M25PE10,
  FL_DEVICE_ST_M25PE20,
  FL_DEVICE_ATMEL_AT25FS010,
  FL_DEVICE_WINBOND_W25X40,
  FL_DEVICE_AMIC_A25L016,
  FL_DEVICE_AMIC_A25L40PT,
  FL_DEVICE_AMIC_A25L40PUM,
  FL_DEVICE_AMIC_A25L80P,
  FL_DEVICE_ATMEL_AT25DF021,
  FL_DEVICE_ATMEL_AT25F512,
  FL_DEVICE_ESMT_F25L004A,
  FL_DEVICE_NUMONYX_M25P10,
  FL_DEVICE_NUMONYX_M25P16,
  FL_DEVICE_NUMONYX_M45P10E,
  FL_DEVICE_SPANSION_S25FL204K,
  FL_DEVICE_SST_SST25VF010,
  FL_DEVICE_SST_SST25VF016,
  FL_DEVICE_SST_SST25VF040,
  FL_DEVICE_WINBOND_W25X10,
  FL_DEVICE_WINBOND_W25X20,
  FL_DEVICE_AMIC_A25L40P,
  FL_DEVICE_MACRONIX_MX25L1005C,
  FL_DEVICE_MICRON_M25P40,
  FL_DEVICE_ALTERA_EPCS1,
};

// There are 16 bytes stored in the data partition.
#define NUM_LED_BYTES 16

// There are 4 LED patterns made up of 4 bytes each.
#define NUM_LED_PATTERNS (NUM_LED_BYTES / 4)

// Utility function for building the led pattern from the led_byte_buffer.
int getPattern(char * led_byte_buffer)
{
  // LED pattern is made up of 4 bytes shifted into an integer value.
  int led_pattern = led_byte_buffer[0] << 24 |
                    led_byte_buffer[1] << 16 |
                    led_byte_buffer[2] << 8 |
                    led_byte_buffer[3];

  return led_pattern;
}

int main(void)
{
  char led_byte_buffer[NUM_LED_BYTES];  // Buffer to hold the 16 bytes read from the data partition.
  int delay = 50;                       // Initial delay 50 ms.
  int led_counter = 0;                  // A counter to count through the led patterns.

  // Connect to the SPI device using the flash library function fl_connectToDevice.
  if(fl_connectToDevice(ports, deviceSpecs, sizeof(deviceSpecs)/sizeof(fl_DeviceSpec)) != 0)
  {
    return 1;
  }

  // Read all 16 bytes from offset 0 in the data partition and store them in 
  // the buffer led_byte_buffer using the flash library function fl_readData.
  if(fl_readData(0, NUM_LED_BYTES, led_byte_buffer) != 0)
  {
    return 1;
  }

   // Disconnect from the SPI device.
  fl_disconnect();

  while(1)
  {
    delay_milliseconds(delay);                          // Wait.
    delay += 1;                                         // Increase the delay so the spinning bar gets slower.
    p32 <: getPattern(&led_byte_buffer[led_counter]);   // Drive the next led pattern.
    led_counter += NUM_LED_PATTERNS;                    // Pick the next pattern.
    if (led_counter == NUM_LED_BYTES)                   // If we are at the last pattern,
    {
      led_counter = 0;                                  // then wrap around.
    }
  }

  return 0;
}
