// Copyright 2011-2025 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.

/**
  * @file      XUD_Main.xc
  * @brief     XMOS USB Device (XUD) Layer
  * @author    Ross Owen
  **/
#include <xs1.h>
#include <print.h>
#include <xclib.h>
#include <platform.h>

#include "xud.h"                 /* External user include file */
#include "XUD_USB_Defines.h"
#include "XUD_Support.h"

#include "XUD_DeviceAttach.h"
#include "XUD_Signalling.h"
#include "XUD_HAL.h"
#include "XUD_TimingDefines.h"

#if (USB_MAX_NUM_EP_IN != 16)
#error USB_MAX_NUM_EP_IN must be 16!
#endif
#if (USB_MAX_NUM_EP_OUT != 16)
#error USB_MAX_NUM_EP_OUT must be 16!
#endif

#if !((XUD_USB_ISO_MAX_TXNS_PER_MICROFRAME == 1) || \
     ((XUD_USB_ISO_MAX_TXNS_PER_MICROFRAME == 2) && defined(__XS3A__)))
#error "XUD_USB_ISO_MAX_TXNS_PER_MICROFRAME must be 1 or 2 (2 only valid on XS3A)"
#endif

#if ((XUD_USB_ISO_EP_MAX_TXN_SIZE) > 1024)
#error "XUD_USB_ISO_EP_MAX_TXN_SIZE must not exceed 1024"
#endif

#if ((XUD_USB_ISO_EP_MAX_TXN_SIZE) % 4)
#error "XUD_USB_ISO_EP_MAX_TXN_SIZE must be a multiple of 4"
#endif

void XUD_UserSuspend();
void XUD_UserResume();
void XUD_PhyReset_User();

#define HS_TX_HANDSHAKE_TIMEOUT (167)
#define FS_TX_HANDSHAKE_TIMEOUT (5000)

/* Global vars for current and desired USB speed */
unsigned g_curSpeed;
unsigned g_desSpeed;
unsigned g_txHandshakeTimeout;

in port flag0_port = PORT_USB_FLAG0; /* For XS3: Mission: RXE, XS2 is configurable and set to RXE in mission mode */
in port flag1_port = PORT_USB_FLAG1; /* For XS3: Mission: RXA, XS2 is configuratble and set to RXA in mission mode*/

/* XS2A has an additonal flag port. In Mission mode this is set to VALID_TOKEN */
#ifdef __XS2A__
in port flag2_port = PORT_USB_FLAG2;
#else
#define flag2_port null
#endif

in buffered port:32 p_usb_clk  = PORT_USB_CLK;
out buffered port:32 p_usb_txd = PORT_USB_TXD;
in  buffered port:32 p_usb_rxd = PORT_USB_RXD;
out port tx_readyout           = PORT_USB_TX_READYOUT;
in port tx_readyin             = PORT_USB_TX_READYIN;
in port rx_rdy                 = PORT_USB_RX_READY;

on USB_TILE: clock tx_usb_clk  = XS1_CLKBLK_4;
on USB_TILE: clock rx_usb_clk  = XS1_CLKBLK_5;

// We use a single array instrad of two here and append epAddr_Ready_setup on the end to save some instructions in the Setup
// token handling code. i.e. what we really want is the following, but's less efficient.
// unsigned epAddr_Ready[USB_MAN_NUM_EP]
// unsigned epAddr_Ready[USB_MAX_NUM_EP_OUT]
unsigned epAddr[USB_MAX_NUM_EP];                        // Used to store the addr of each EP in ep_info array
unsigned epAddr_Ready[USB_MAX_NUM_EP + USB_MAX_NUM_EP_OUT];    // Used by the EP to mark itself as ready, essentially same as epAddr with 0 entries.

XUD_chan epChans0[USB_MAX_NUM_EP];

XUD_ep_info ep_info[USB_MAX_NUM_EP];

/* Location to store stack pointer (required for interrupt handler) */
unsigned SavedSp;

/* Tables storing if EP's are signed up to bus state updates */
int epStatFlagTableIn[USB_MAX_NUM_EP_IN];
int epStatFlagTableOut[USB_MAX_NUM_EP_OUT];

unsigned sentReset = 0;

#define RESET_TIME_us               (5)
#define RESET_TIME                  (RESET_TIME_us * PLATFORM_REFERENCE_MHZ)

extern unsigned XUD_LLD_IoLoop(
                            in buffered port:32 rxd_port,
                            in port rxa_port,
                            out buffered port:32 txd_port,
                            in port rxe_port, in port ?valtok_port,
                            XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[], XUD_chan epAddrReady_local[],
                            int  epCount, chanend? c_sof) ;


#if (XUD_OPT_SOFTCRC5 == 1)
extern unsigned char crc5Table[2048];
extern unsigned char crc5Table_Addr[2048];

void XUD_SetCrcTableAddr(unsigned addr);
#endif

static int one = 1;

#pragma unsafe arrays
void SendBusStateToEps(XUD_chan c[], XUD_chan epAddrReady_local[], XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[], int nOut, int nIn, unsigned token)
{
    for(int i = 0; i < nOut; i++)
    {
        if(epTypeTableOut[i] != XUD_EPTYPE_DIS && epStatFlagTableOut[i])
        {
            /* Set EP busUpdate flag. EP uses this to check if it missed a reset before setting ready.
             * This is required since a "non-ready" EP will not have received its notifcation, it may
             * subsequently re-mark itself ready causing XUD to send out of date data */
            ep_info[i].busUpdate = 1;

            /* Clear EP ready. Note. small race since EP might set ready after XUD sets busUpdate to 1
             * but this should be caught in time (EP gets CT) */
            epAddrReady_local[i] = 0;
            epAddrReady_local[i+ USB_MAX_NUM_EP] = 0;
            XUD_Sup_outct(c[i], token);
        }
    }
    for(int i = 0; i < nIn; i++)
    {
        if(epTypeTableIn[i] != XUD_EPTYPE_DIS && epStatFlagTableIn[i])
        {
            ep_info[i + USB_MAX_NUM_EP_OUT].busUpdate = 1;
            epAddrReady_local[i + USB_MAX_NUM_EP_OUT] = 0;
            XUD_Sup_outct(c[i + USB_MAX_NUM_EP_OUT], token);
        }
    }
}

#pragma unsafe arrays
static void SendSpeed(XUD_chan c[], XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[], int nOut, int nIn, int speed)
{
    for(int i = 0; i < nOut; i++)
    {
        if(epTypeTableOut[i] != XUD_EPTYPE_DIS && epStatFlagTableOut[i])
        {
            XUD_Sup_outuint(c[i], speed);
        }
    }
    for(int i = 0; i < nIn; i++)
    {
        if(epTypeTableIn[i] != XUD_EPTYPE_DIS && epStatFlagTableIn[i])
        {
            XUD_Sup_outuint(c[i + USB_MAX_NUM_EP_OUT], speed);
        }
    }
}

#pragma unsafe arrays
void GetCTFromEps(XUD_chan c[], XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[], int nOut, int nIn)
{
    for(int i = 0; i < nOut; i++)
    {
        if(epTypeTableOut[i] != XUD_EPTYPE_DIS && epStatFlagTableOut[i])
        {
            XUD_Sup_inct(c[i]);
        }
    }
    for(int i = 0; i < nIn; i++)
    {
        if(epTypeTableIn[i] != XUD_EPTYPE_DIS && epStatFlagTableIn[i])
        {
            XUD_Sup_inct(c[i + USB_MAX_NUM_EP_OUT]);
        }
    }
}

// Main XUD loop
static int XUD_Manager_loop(XUD_chan epChans0_local[], XUD_chan epAddrReady_local[],  chanend ?c_sof, XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[], int noEpOut, int noEpIn, XUD_PwrConfig pwrConfig)
{
    int reset = 1;            /* Flag for if device is returning from a reset */

    /* Make sure ports are on and reset port states */
    set_port_use_on(p_usb_clk);
    set_port_use_on(p_usb_txd);
    set_port_use_on(p_usb_rxd);
    set_port_use_on(flag0_port);
    set_port_use_on(flag1_port);
#if defined(__XS2A__)
    /* Extra flag port in XS2 */
    set_port_use_on(flag2_port);
#endif

#if !defined(__XS2A__)
    /* XS3 settings */
    #ifndef XUD_CORE_CLOCK
        #error XUD_CORE_CLOCK not defined (in MHz)
    #endif

    /* Sim settings are currenly different from silicon due to differences in sim model and silicon behaviour - see https://github.com/xmos/lib_xud/issues/427*/
    #ifdef XUD_SIM_XSIM
        #if (XUD_CORE_CLOCK >= 800)
            #define RX_RISE_DELAY 6 // Si setting
            #define TX_RISE_DELAY 6 // Only 6 works
            #define TX_FALL_DELAY 5 // 5 or less works
        #elif (XUD_CORE_CLOCK >= 700)
            #define RX_RISE_DELAY 5 // Si setting
            #define TX_RISE_DELAY 5 // Only 5 works
            #define TX_FALL_DELAY 4 // 4 or less works
        #elif (XUD_CORE_CLOCK >= 600)
            #define RX_RISE_DELAY 5 // Si setting
            #define TX_RISE_DELAY 4 // 4 or 5 works
            #define TX_FALL_DELAY 2 // 2 or less works
        #elif (XUD_CORE_CLOCK >= 500)
            #define RX_RISE_DELAY 4 // Si setting
            #define TX_RISE_DELAY 4 // Only 4 works
            #define TX_FALL_DELAY 0 // Only 0 works
        #else
            #error XUD_CORE_CLOCK must be >= 500
        #endif
    #else
    /* See https://xmosjira.atlassian.net/wiki/spaces/~870418189/pages/4627398657/on-die+USB+PHY+timings */
        #if (XUD_CORE_CLOCK >= 800)
            #define RX_RISE_DELAY 6
            #define TX_RISE_DELAY 3
            #define TX_FALL_DELAY 6
        #elif (XUD_CORE_CLOCK >= 700)
            #define RX_RISE_DELAY 5
            #define TX_RISE_DELAY 3
            #define TX_FALL_DELAY 5
        #elif (XUD_CORE_CLOCK >= 600)
            #define RX_RISE_DELAY 5
            #define TX_RISE_DELAY 3
            #define TX_FALL_DELAY 4
        #elif (XUD_CORE_CLOCK >= 500)
            #define RX_RISE_DELAY 4
            #define TX_RISE_DELAY 2
            #define TX_FALL_DELAY 2
        #elif (XUD_CORE_CLOCK >= 400)
            #define RX_RISE_DELAY 3
            #define TX_RISE_DELAY 2
            #define TX_FALL_DELAY 2
        #else
            #error XUD_CORE_CLOCK must be >= 400
        #endif
    #endif
#else /* Settings for XS2 */
    #define RX_RISE_DELAY 1
    #define RX_FALL_DELAY 5
    #define TX_RISE_DELAY 5
    #define TX_FALL_DELAY 1
    #define RX_ACTIVE_PAD_DELAY 2
#endif

    // Handshaken ports need USB clock
    configure_clock_src(tx_usb_clk, p_usb_clk);
    configure_clock_src(rx_usb_clk, p_usb_clk);

#if defined(__XS2A__)
    // This, along with the following delays, forces the clock
    // to the ports to be effectively controlled by the
    // previous usb clock edges
    set_port_inv(p_usb_clk);

    // This delay controls the capture of rdy
    set_clock_rise_delay(tx_usb_clk, TX_RISE_DELAY);

    // This delay controls the launch of data.
    set_clock_fall_delay(tx_usb_clk, TX_FALL_DELAY);

    // This delay the capture of the rdyIn and data.
    set_clock_rise_delay(rx_usb_clk, RX_RISE_DELAY);
    set_clock_fall_delay(rx_usb_clk, RX_FALL_DELAY);

    set_pad_delay(flag1_port, RX_ACTIVE_PAD_DELAY);

#else
    /* Settings for xcore.ai */
    /* We are using non-inverted clock for xcore.ai with a significant TX_FALL_DELAY as per
    http://cognidox.xmos.local/cgi-perl/part-details?partnum=XM-015222-PS */

    // This delay controls the capture of rdy. We need to get this before the earliest rising edge.
    set_clock_rise_delay(tx_usb_clk, TX_RISE_DELAY);

    // This delay controls the launch of data and strobe.
    set_clock_fall_delay(tx_usb_clk, TX_FALL_DELAY);

    // This delays the capture of the rdyIn and data. This is delayed to center the window.
    set_clock_rise_delay(rx_usb_clk, RX_RISE_DELAY);

#endif // __XS2A__

    start_clock(tx_usb_clk);
    start_clock(rx_usb_clk);

 	configure_out_port_handshake(p_usb_txd, tx_readyin, tx_readyout, tx_usb_clk, 0);
  	configure_in_port_strobed_slave(p_usb_rxd, rx_rdy, rx_usb_clk);

    /* Clock RxA port from USB clock - helps fall event */
    configure_in_port(flag1_port, rx_usb_clk);

    unsigned noExit = 1;

    while(noExit)
    {
        unsigned settings[] = {0};

        /* Enable USB funcitonality in the device */
        XUD_HAL_EnableUsb(pwrConfig);

        while(1)
        {
            {
                /* Wait for VBUS before enabling pull-up. The USB Spec (page 150) allows 100ms
                 * between vbus valid and signalling attach */
                if(pwrConfig == XUD_PWR_SELF)
                {
                    while(1)
                    {
                        unsigned time;
                        timer t;

                        if(XUD_HAL_GetVBusState())
                        {
                            break;
                        }
                        t :> time;
                        time += (200 * PLATFORM_REFERENCE_MHZ); // 200us poll
                        t when timerafter(time):> void;
                    }
                }

                /* Go into full speed mode: XcvrSelect and Term Select (and suspend) high */
                XUD_HAL_EnterMode_PeripheralFullSpeed();

                /* Setup flags for power signalling - i.e. J/K/SE0 line state*/
                XUD_HAL_Mode_Signalling();

                if (one)
                {
#if defined(XUD_BYPASS_CONNECT)
                    reset = 1;
#else
                    reset = XUD_Init();
#endif
                    one = 0;
                }
                else
                {
                    timer t; unsigned time;
                    t :> time;
                    t when timerafter(time + SUSPEND_T_WTWRSTHS_ticks) :> int _;// T_WTRSTHS: 100-875us

                    /* Sample line state and check for reset (or suspend) */
                    XUD_LineState_t ls = XUD_HAL_GetLineState();
                    if(ls == XUD_LINESTATE_SE0)
                        reset = 1;
                    else
                        reset = 0;
                }

                /* Handle suspend */
                if(!reset)
                {
                    /* Run user suspend code */
                    XUD_UserSuspend();

                    /* Run suspend code, returns 1 if reset from suspend, 0 for resume, -1 for invalid vbus */
                    reset = XUD_Suspend(pwrConfig, epChans0_local, epAddrReady_local, epTypeTableOut, epTypeTableIn, noEpOut, noEpIn);

                    if((pwrConfig == XUD_PWR_SELF) && (reset==-1))
                    {
                        /* Lost VBUS */
                        continue;
                    }

                    /* Run user resume code */
                    XUD_UserResume();
                }

                /* Handle bus reset */
                if(reset == 1)
                {
                    if(!sentReset)
                    {
                        SendBusStateToEps(epChans0_local, epAddrReady_local, epTypeTableOut, epTypeTableIn, noEpOut, noEpIn, XUD_BUS_RESET);
                        sentReset = 1;
                    }

                    /* Reset the OUT ep structures */
                    for(int i = 0; i< noEpOut; i++)
                    {
#if !defined(__XS2A__)
                        ep_info[i].pid = USB_PIDn_DATA0;
#else
                        ep_info[i].pid = USB_PID_DATA0;
#endif
                    }

                    /* Reset in the ep structures */
#if (XUD_USB_ISO_MAX_TXNS_PER_MICROFRAME > 1)
                    for(int i = 0; i< noEpIn; i++)
                    {
                        if(ep_info[USB_MAX_NUM_EP_OUT+i].epType != XUD_EPTYPE_ISO)
                        {
                            ep_info[USB_MAX_NUM_EP_OUT+i].pid = USB_PIDn_DATA0;
                        }
                    }
#else
                    for(int i = 0; i< noEpIn; i++)
                    {
                        ep_info[USB_MAX_NUM_EP_OUT+i].pid = USB_PIDn_DATA0;
                    }
#endif

                    /* Set default device address - note, for normal operation this is 0, but can be other values for testing */
                    XUD_HAL_SetDeviceAddress(XUD_STARTUP_ADDRESS);

#ifdef XUD_BYPASS_RESET
                    if(XUD_TEST_SPEED == XUD_SPEED_HS)
                    {
                        g_curSpeed = XUD_SPEED_HS;
                        g_txHandshakeTimeout = HS_TX_HANDSHAKE_TIMEOUT;
                        XUD_HAL_EnterMode_PeripheralHighSpeed();
                    }
                    else
                    {
                        g_curSpeed = XUD_SPEED_FS;
                        g_txHandshakeTimeout = FS_TX_HANDSHAKE_TIMEOUT;
                        XUD_HAL_EnterMode_PeripheralFullSpeed(); //Technically not required since we should already be in FS mode..
                    }
#else
                    if(g_desSpeed == XUD_SPEED_HS)
                    {
                        unsigned tmp = 0;
                        tmp = XUD_DeviceAttachHS(pwrConfig);

                        if(tmp == -1)
                        {
                            XUD_UserSuspend();
                            continue;
                        }
                        else if (!tmp)
                        {
                            /* HS handshake fail, mark as running in FS */
                            g_curSpeed = XUD_SPEED_FS;
                            g_txHandshakeTimeout = FS_TX_HANDSHAKE_TIMEOUT;
                        }
                        else
                        {
                            g_curSpeed = XUD_SPEED_HS;
                            g_txHandshakeTimeout = HS_TX_HANDSHAKE_TIMEOUT;
                        }
                    }
                    else
                    {
                        g_curSpeed = XUD_SPEED_FS;
                        g_txHandshakeTimeout = FS_TX_HANDSHAKE_TIMEOUT;
                    }
#endif

                    /* Send speed to EPs */
                    SendSpeed(epChans0_local, epTypeTableOut, epTypeTableIn, noEpOut, noEpIn, g_curSpeed);
                    sentReset=0;
                }
            }

            XUD_HAL_Mode_DataTransfer();

            if(XUD_THREAD_MODE_FAST_EN)
            {
                set_thread_fast_mode_on();
            }
            /* Run main IO loop */
            /* flag0: Rx Error
               flag1: Rx Active
               flag2: Null / Valid Token  */
            noExit = XUD_LLD_IoLoop(p_usb_rxd, flag1_port, p_usb_txd, flag0_port, flag2_port, epTypeTableOut, epTypeTableIn, epAddrReady_local, noEpOut, c_sof);


            if(XUD_THREAD_MODE_FAST_EN)
            {
                set_thread_fast_mode_off();
            }
            // Reset ep->saved_frame to 0 to allow sof counting to resume after suspend
            for(int i = 0; i< noEpOut; i++)
            {
                ep_info[i].saved_frame = 0;
            }

            /* Reset in the ep structures */
            for(int i = 0; i< noEpIn; i++)
            {
                ep_info[USB_MAX_NUM_EP_OUT+i].saved_frame = 0;
            }

            if(!noExit)
                break;
        }
    }

    /* TODO stop clock blocks */

    /* Turn ports off */
    set_port_use_off(p_usb_txd);
    set_port_use_off(p_usb_rxd);
    set_port_use_off(flag0_port);
    set_port_use_off(flag1_port);
#ifdef __XS2A__
    set_port_use_off(flag2_port);
#endif
    set_port_use_off(p_usb_clk);
    return 0;
}

void _userTrapHandleRegister(void);

#pragma unsafe arrays
static void _XUD_drain(chanend chans[], int n, int op, XUD_EpType epTypeTable[])
{
    for(int i = 0; i < n; i++)
    {
        if(epTypeTable[i] != XUD_EPTYPE_DIS)
        {
            switch(op)
            {
                case 0:
                    outct(chans[i], XUD_BUS_KILL);
                    break;
                case 1:
                    outct(chans[i], XS1_CT_END);
                    while (!testct(chans[i]))
                        inuchar(chans[i]);
                    chkct(chans[i], XS1_CT_END);
                     break;
            }
        }
    }
}

#pragma unsafe arrays
void SetupEndpoints(chanend c_ep_out[], int noEpOut, chanend c_ep_in[], int noEpIn, XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[])
{

    for(int i = 0; i < USB_MAX_NUM_EP_OUT; i++)
    {
        unsigned x;
        epAddr_Ready[i] = 0;
        epAddr_Ready[i+USB_MAX_NUM_EP] = 0; //epAddr_Ready_Setup
        ep_info[i].epAddress = i;
        ep_info[i].busUpdate = 0;
        ep_info[i].current_transaction = 0;

        /* Mark all EP's as halted, we might later clear this if the EP is in use */
        ep_info[i].halted = USB_PIDn_STALL;

        asm("ldaw %0, %1[%2]":"=r"(x):"r"(ep_info),"r"(i*sizeof(XUD_ep_info)/sizeof(unsigned)));
        epAddr[i] = x;
    }

    for(int i = 0; i < USB_MAX_NUM_EP_IN; i++)
    {
        unsigned x;
        ep_info[i].epAddress = i;
        epAddr_Ready[USB_MAX_NUM_EP_OUT+i] = 0;
        ep_info[USB_MAX_NUM_EP_OUT+i].epAddress = (i | 0x80);
        ep_info[USB_MAX_NUM_EP_OUT+i].busUpdate = 0;
        ep_info[USB_MAX_NUM_EP_OUT+i].halted = USB_PIDn_STALL;
        ep_info[USB_MAX_NUM_EP_OUT+i].current_transaction = 0;

        asm("ldaw %0, %1[%2]":"=r"(x):"r"(ep_info),"r"((USB_MAX_NUM_EP_OUT+i)*sizeof(XUD_ep_info)/sizeof(unsigned)));
        epAddr[USB_MAX_NUM_EP_OUT+i] = x;
    }

    /* Populate arrays of channels and status flag tables */
    /* Note, if the epTypeTables don't match the provided size there could be trouble.. */
    for(int i = 0; i < noEpOut; i++)
    {
        if(epTypeTableOut[i] != XUD_EPTYPE_DIS)
        {
            unsigned x;
            epChans0[i] = XUD_Sup_GetResourceId(c_ep_out[i]);

            asm("ldaw %0, %1[%2]":"=r"(x):"r"(epAddr_Ready),"r"(i));
            ep_info[i].array_ptr = x;
            ep_info[i].saved_array_ptr = 0;

            asm("ldaw %0, %1[%2]":"=r"(x):"r"(epAddr_Ready),"r"(i+USB_MAX_NUM_EP)); //epAddr_Ready_Setup
            ep_info[i].array_ptr_setup = x;

            asm("mov %0, %1":"=r"(x):"r"(c_ep_out[i]));
            ep_info[i].xud_chanend = x;

            asm("getd %0, res[%1]":"=r"(x):"r"(c_ep_out[i]));
            ep_info[i].client_chanend = x;

            epStatFlagTableOut[i] = epTypeTableOut[i] & XUD_STATUS_ENABLE;
            epTypeTableOut[i] = epTypeTableOut[i] & 0x7FFFFFFF;

            ep_info[i].epType = epTypeTableOut[i];
            ep_info[i].halted = USB_PIDn_NAK;      // Mark EP as not halted
            ep_info[i].remained = 0;
            ep_info[i].saved_frame = 0;
            ep_info[i].max_len = XUD_USB_ISO_EP_MAX_TXN_SIZE;
            ep_info[i].out_err_flag = 0;
#if !defined(__XS2A__)
            ep_info[i].pid = USB_PIDn_DATA0;
#else
            ep_info[i].pid = USB_PID_DATA0;
#endif
            asm("ldaw %0, %1[%2]":"=r"(x):"r"(ep_info),"r"(i*sizeof(XUD_ep_info)/sizeof(unsigned)));
            outuint(c_ep_out[i], x);
        }
    }

    for(int i = 0; i< noEpIn; i++)
    {
        if(epTypeTableIn[i] != XUD_EPTYPE_DIS)
        {
            int x;
            epChans0[i+USB_MAX_NUM_EP_OUT] = XUD_Sup_GetResourceId(c_ep_in[i]);

            asm("ldaw %0, %1[%2]":"=r"(x):"r"(epAddr_Ready),"r"(USB_MAX_NUM_EP_OUT+i));
            ep_info[USB_MAX_NUM_EP_OUT+i].array_ptr = x;
            ep_info[USB_MAX_NUM_EP_OUT+i].saved_array_ptr = 0;

            asm("mov %0, %1":"=r"(x):"r"(c_ep_in[i]));
            ep_info[USB_MAX_NUM_EP_OUT+i].xud_chanend = x;

            asm("getd %0, res[%1]":"=r"(x):"r"(c_ep_in[i]));
            ep_info[USB_MAX_NUM_EP_OUT+i].client_chanend = x;

            ep_info[USB_MAX_NUM_EP_OUT+i].pid = USB_PIDn_DATA0;

            epStatFlagTableIn[i] = epTypeTableIn[i] & XUD_STATUS_ENABLE;
            epTypeTableIn[i] = epTypeTableIn[i] & 0x7FFFFFFF;

            ep_info[USB_MAX_NUM_EP_OUT+i].epType = epTypeTableIn[i];

            ep_info[USB_MAX_NUM_EP_OUT+i].max_len = XUD_USB_ISO_EP_MAX_TXN_SIZE;

            ep_info[USB_MAX_NUM_EP_OUT+i].halted = 0;    // Mark EP as not halted

            ep_info[USB_MAX_NUM_EP_OUT+i].remained = 0;
            ep_info[i].num_transactions = 1;
            ep_info[USB_MAX_NUM_EP_OUT+i].saved_frame = 0;
            ep_info[USB_MAX_NUM_EP_OUT+i].out_err_flag = 0;

            asm("ldaw %0, %1[%2]":"=r"(x):"r"(ep_info),"r"((USB_MAX_NUM_EP_OUT+i)*sizeof(XUD_ep_info)/sizeof(unsigned)));
            outuint(c_ep_in[i], x);
        }
    }

    /* EpTypeTable Checks.  Note, currently this is not too crucial since we only really care if the EP is ISO or not */

    /* Check for control on IN/OUT 0 */
    if(epTypeTableOut[0] != XUD_EPTYPE_CTL || epTypeTableIn[0] != XUD_EPTYPE_CTL)
    {
        __builtin_trap();
    }
}


#pragma unsafe arrays
int XUD_Main(chanend c_ep_out[], int noEpOut,
                chanend c_ep_in[], int noEpIn,
                chanend ?c_sof,
                XUD_EpType epTypeTableOut[], XUD_EpType epTypeTableIn[],
                XUD_BusSpeed_t speed, XUD_PwrConfig pwrConfig)
{
    g_desSpeed = speed;

    SetupEndpoints(c_ep_out, noEpOut, c_ep_in, noEpIn, epTypeTableOut, epTypeTableIn);

    /* Run the main XUD loop */
    XUD_Manager_loop(epChans0, epAddr_Ready, c_sof, epTypeTableOut, epTypeTableIn, noEpOut, noEpIn, pwrConfig);

    // Need to close, drain, and check - three stages.
    for(int i = 0; i < 2; i++)
    {
        _XUD_drain(c_ep_out, noEpOut, i, epTypeTableOut);  // On all inputs
        _XUD_drain(c_ep_in, noEpIn, i, epTypeTableIn);     // On all output
    }

    return 0;
}


